/*
 * Decompiled with CFR 0.152.
 */
package dev.tr7zw.lib.sentry;

import dev.tr7zw.lib.sentry.CircularFifoQueue;
import dev.tr7zw.lib.sentry.DataCategory;
import dev.tr7zw.lib.sentry.Hint;
import dev.tr7zw.lib.sentry.ILogger;
import dev.tr7zw.lib.sentry.IScopes;
import dev.tr7zw.lib.sentry.SentryLevel;
import dev.tr7zw.lib.sentry.SynchronizedQueue;
import dev.tr7zw.lib.sentry.hints.Cached;
import dev.tr7zw.lib.sentry.hints.Enqueable;
import dev.tr7zw.lib.sentry.hints.Flushable;
import dev.tr7zw.lib.sentry.hints.Retryable;
import dev.tr7zw.lib.sentry.hints.SubmissionResult;
import dev.tr7zw.lib.sentry.transport.RateLimiter;
import dev.tr7zw.lib.sentry.util.HintUtils;
import java.io.File;
import java.util.Queue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

abstract class DirectoryProcessor {
    private static final long ENVELOPE_PROCESSING_DELAY = 100L;
    @NotNull
    private final IScopes scopes;
    @NotNull
    private final ILogger logger;
    private final long flushTimeoutMillis;
    private final Queue<String> processedEnvelopes;

    DirectoryProcessor(@NotNull IScopes scopes, @NotNull ILogger logger, long flushTimeoutMillis, int maxQueueSize) {
        this.scopes = scopes;
        this.logger = logger;
        this.flushTimeoutMillis = flushTimeoutMillis;
        this.processedEnvelopes = SynchronizedQueue.synchronizedQueue(new CircularFifoQueue(maxQueueSize));
    }

    public void processDirectory(@NotNull File directory) {
        try {
            this.logger.log(SentryLevel.DEBUG, "Processing dir. %s", directory.getAbsolutePath());
            File[] filteredListFiles = directory.listFiles((d, name) -> this.isRelevantFileName(name));
            if (filteredListFiles == null) {
                this.logger.log(SentryLevel.ERROR, "Cache dir %s is null or is not a directory.", directory.getAbsolutePath());
                return;
            }
            this.logger.log(SentryLevel.DEBUG, "Processing %d items from cache dir %s", filteredListFiles.length, directory.getAbsolutePath());
            for (File file : filteredListFiles) {
                if (!file.isFile()) {
                    this.logger.log(SentryLevel.DEBUG, "File %s is not a File.", file.getAbsolutePath());
                    continue;
                }
                String filePath = file.getAbsolutePath();
                if (this.processedEnvelopes.contains(filePath)) {
                    this.logger.log(SentryLevel.DEBUG, "File '%s' has already been processed so it will not be processed again.", filePath);
                    continue;
                }
                @Nullable RateLimiter rateLimiter = this.scopes.getRateLimiter();
                if (rateLimiter != null && rateLimiter.isActiveForCategory(DataCategory.All)) {
                    this.logger.log(SentryLevel.INFO, "DirectoryProcessor, rate limiting active.", new Object[0]);
                    return;
                }
                this.logger.log(SentryLevel.DEBUG, "Processing file: %s", filePath);
                SendCachedEnvelopeHint cachedHint = new SendCachedEnvelopeHint(this.flushTimeoutMillis, this.logger, filePath, this.processedEnvelopes);
                Hint hint = HintUtils.createWithTypeCheckHint(cachedHint);
                this.processFile(file, hint);
                Thread.sleep(100L);
            }
        }
        catch (Throwable e) {
            this.logger.log(SentryLevel.ERROR, e, "Failed processing '%s'", directory.getAbsolutePath());
        }
    }

    protected abstract void processFile(@NotNull File var1, @NotNull Hint var2);

    protected abstract boolean isRelevantFileName(String var1);

    private static final class SendCachedEnvelopeHint
    implements Cached,
    Retryable,
    SubmissionResult,
    Flushable,
    Enqueable {
        boolean retry = false;
        boolean succeeded = false;
        private final CountDownLatch latch;
        private final long flushTimeoutMillis;
        @NotNull
        private final ILogger logger;
        @NotNull
        private final String filePath;
        @NotNull
        private final Queue<String> processedEnvelopes;

        public SendCachedEnvelopeHint(long flushTimeoutMillis, @NotNull ILogger logger, @NotNull String filePath, @NotNull Queue<String> processedEnvelopes) {
            this.flushTimeoutMillis = flushTimeoutMillis;
            this.filePath = filePath;
            this.processedEnvelopes = processedEnvelopes;
            this.latch = new CountDownLatch(1);
            this.logger = logger;
        }

        @Override
        public boolean isRetry() {
            return this.retry;
        }

        @Override
        public void setRetry(boolean retry) {
            this.retry = retry;
        }

        @Override
        public boolean waitFlush() {
            try {
                return this.latch.await(this.flushTimeoutMillis, TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                this.logger.log(SentryLevel.ERROR, "Exception while awaiting on lock.", e);
                return false;
            }
        }

        @Override
        public void setResult(boolean succeeded) {
            this.succeeded = succeeded;
            this.latch.countDown();
        }

        @Override
        public boolean isSuccess() {
            return this.succeeded;
        }

        @Override
        public void markEnqueued() {
            this.processedEnvelopes.add(this.filePath);
        }
    }
}

